  ' -----------------------------------------------------------
  ' DRV8825 stepper driving program.bas
  ' A program for Micromite to control the operation of a bipolar
  ' stepper motor via a DRV8825 based driver module.
  ' 
  ' Written by Jim Rowe for Silicon Chip.
  ' Last revision 8/11/2018 at 10:00 am
  '
  ' Notes:
  ' 1. The DRV8825 device has an inbuilt indexing controller, so
  ' control of the motor is simplified in terms of the software. Most
  ' control functionality is via the DIR(direction) pin driven from pin 9
  ' of the Micromite, the STP(step) pin driven from pin 10, the SLP(Sleep-bar) 
  ' and RST(reset-bar) pins driven from pin 16 and the EN(enable-bar)
  ' pin driven from pin 22.
  ' 2. The actual stepping configuration of the indexing controller in
  ' the DRV8825 is programmed via pins M0, M1 and M2, driven here from
  ' Micromite pins 21, 18 and 17 respectively. These are the main options:
  '    M0     M1    M2    STEPPING MODE
  '     0      0     0    Full stepping
  '     1      0     0    1/2 stepping
  '     0      1     0    1/4 stepping
  '     1      1     0    8 microsteps per step
  '     0      0     1    16 microsteps per step
  '     1      0     1    32 microsteps per step.
  ' 3. The two GND pins of the DRV8825 module should be connected together,
  ' and also to one GND pin of the Micromite - as well as being connected to
  ' the negative side of the motor supply.
  ' 4. The motor supply voltage VMA should have an electrolytic capacitor of
  ' at least 100uF connected as close as possible to the module and between
  ' the VMA and GND pins.
  ' ----------------------------------------------------------
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  
  DIM AS INTEGER IntFlag = 0      ' flag to show when LCD screen is touched
  DIM AS INTEGER ExFlag = 0       ' flag for exiting from a screen
  DIM AS INTEGER StpFlag = 0      ' flag for stepping (1) or stopped (0)
  DIM AS INTEGER SwingFlg = 0     ' flag for FWD/REV swing mode
    
  DIM AS INTEGER StpsperRev = 200 ' define steps per rev for motor in use
                                  ' (change to suit your motor)
  DIM AS INTEGER Steps2Go = 1     ' number of steps required
  DIM AS INTEGER StpMult = 1      ' step multiplier for multi-step modes
  DIM AS INTEGER Ctr = 1          ' loop counter
  
  CONST TRUE = 1
  CONST FALSE = 0
  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  
  SETPIN 15, INTL, TchInt ' call TchInt to set flag when screen touched
  
  SETPIN 9, DOUT          ' declare pin 9 a digital output (for DIR)
  SETPIN 10, DOUT         ' and pin 10 a digital output (for STP)
  SETPIN 16, DOUT         ' also pin 16 (for SLP)
  SETPIN 17, DOUT         ' pin 17 (for M2)
  SETPIN 18, DOUT         ' pin 18 (for M1)
  SETPIN 21, DOUT         ' pin 21 (for M0)
  SETPIN 22, DOUT         ' and finally pin 22 (for EN)
  
  PIN(9) = 0    ' set pin 9 low to initialise for forward stepping
  PIN(10) = 0   ' and pin 10 low for no step
  PIN(22) = 1   ' and pin 22 high to disable DRV8825 outputs
  PIN(21) = 0   ' and the mode pins all low for full stepping
  PIN(18) = 0
  PIN(17) = 0
  PIN(16) = 1   ' and pin 16 high so the DRV8825 is no longer sleeping
                ' or reset
  ' *****************************************************************
  ' main program starts here
  MainScrn:    ShowOpenScreen    'show the opening/control screen
  DO
    IF IntFlag = 1 THEN
      CheckBtn   ' if screen was touched, go check & respond
      IF StpFlag = 1 THEN       ' now if StpFlag has been set (i.e., START)
        PIN(22) = 0     ' drop pin 22 to enable the DRV8825 outputs
        IF SwingFlg = 0 THEN
          StpLoop       ' FOR/REV flag not set, so just go to send step(s)
        ELSE
          FwdRevLoop    ' FOR/REV flag is set, so go send FOR/REV steps
        ENDIF
      ENDIF
      SwingFlg = 0      ' reset flags
      StpFlag = 0
      PIN(22) = 1       ' raise pin 22 to disable DRV8825 outputs
    ENDIF
  LOOP
END ' end of main part of program, subroutines follow
  
  ' ***************************************************************** 
  ' subroutine to set IntFlg when screen is touched
SUB TchInt
  IntFlag = 1
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to send one or more steps in same direction
SUB StpLoop
  FOR Ctr = 1 TO Steps2Go STEP 1
  SndaStep
  IF IntFlag = 1 THEN ' in case STOP was touched
    IntFlag = 0       ' just clear flag
    EXIT SUB          ' and exit from the loop
  ENDIF
  NEXT
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to send multiple steps in one direction, then in reverse
  ' direction
SUB FwdRevLoop
  Pin(9) = 0    ' set for forward stepping
  StpLoop       ' and go send one sequence of steps
  PIN(9) = 1    ' then set for reverse stepping
  StpLoop       ' and send another sequence of steps
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to actually send a step pulse (high for 10ms, then low
  ' for the same period -- giving a step frequency of 50Hz)
SUB SndaStep
  PIN(10) = 1   ' send a 1 to the STP pin
  PAUSE 10      ' pause for 10ms
  PIN(10) = 0   ' then send a 0 to end the pulse
  PAUSE 10      ' and pause again for 10ms
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to show the opening/control screen
SUB ShowOpenScreen
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT MM.HRes/2, MM.VRes/16, "SILICON CHIP", CM, 1, 2, Red, Black
  TEXT MM.HRes/2, MM.VRes*3/16, "Stepper Motor Controller",CM, 1, 1, White, Black
  
  RBOX 4, MM.VRes/4, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Green
  TEXT MM.HRes/4, MM.VRes*3/8, "SET FUNCT", CM, 1, 2, Black, Green

  RBOX MM.HRes/2, MM.VRes/4, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*3/8, "SET MODE", CM, 1, 2, Red, Bone
   
  RBOX 4, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Red
  TEXT MM.HRes/4, MM.VRes*5/8, "< DIR", CM, 1, 2, White, Red 
   
  RBOX 4, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes/4, MM.VRes*7/8, "START", CM, 1, 3, Black, Bone 
   
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Red
  TEXT MM.HRes*3/4, MM.VRes*5/8, "DIR >", CM, 1,2, White, Red 
   
  RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*7/8, "STOP", CM, 1,3, Black, Bone    
END SUB
  ' ----------------------------------------------------------------------
  ' subroutine called when opening screen is touched, to check if it has
  ' been touched on either the SET FUNCT, SET MODE, <DIR, DIR>, START
  ' or STOP buttons on the LCD (or not, in which case it just returns)
SUB CheckBtn
  IntFlag = 0       ' first clear interrupt flag
  IF TOUCH(Y) < MM.VRes/4 THEN
    EXIT SUB        ' no valid button touch, so just exit
  END IF
  IF TOUCH(X) < MM.HRes/2 THEN    ' if touch was on lower LH side
    SELECT CASE TOUCH(Y)
      CASE < MM.VRes/2  ' SET FUNCT button touched, so 
        SelFunct        ' go to select function screen
        GOTO MainScrn   ' then return to opening screen
      CASE MM.VRes/2 TO MM.VRes*3/4 ' <DIR button touched, so 
        PIN(9) = 1      ' set pin 9 (DIR) for reverse rotation
      CASE > MM.VRes*3/4  ' START button touched, so 
        StpFlag = 1     ' set StpFlag to 1
    end select
  ELSE    ' touch must have been on lower RH side
    SELECT CASE TOUCH(Y)
      CASE < MM.VRes/2  ' SET MODE button touched, so
        SelMode         ' go to select mode screen
        GOTO MainScrn   ' then return to opening screen
      CASE MM.VRes/2 TO MM.VRes*3/4 ' DIR> button touched, so 
        PIN(9) = 0      ' set pin 9 (DIR) for forward rotation
      CASE > MM.VRes*3/4  ' STOP button touched, so
        StpFlag = 0     ' clear StpFlag
    end select
  END IF
END SUB             ' before returning  
  ' -------------------------------------------------------------------
  ' subroutine to show SelFunction screen & allow function selection
SUB SelFunct
  ShowSFscreen    ' first go show the SF screen
  ExFlag = 0      ' clear exit flag
  DO
    IF IntFlag = 1 THEN  ' if screen touched,
      CheckFn            ' go check & respond
      IF ExFlag = 1 THEN GOTO MainScrn
    ENDIF    
  LOOP
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to show SF screen
SUB ShowSFscreen
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT MM.HRes/2, MM.VRes/16, "SELECT DESIRED", CM, 1, 2, White, Black
  TEXT MM.HRes/2, MM.VRes*3/16, "STEP FUNCTION:",CM, 1, 2, White, Black
  
  RBOX 4, MM.VRes/4, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes/4, MM.VRes*3/8, "SINGLE", CM, 1, 2, Black, Bone

  RBOX MM.HRes/2, MM.VRes/4, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*3/8, "CONTIN", CM, 1, 2, Black, Bone
   
  RBOX 4, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes/4, MM.VRes*5/8, "1/2 REV", CM, 1, 2, Black, Bone 
   
  RBOX 4, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes/4, MM.VRes*7/8, "FWD-REV", CM, 1, 2, Black, Bone 
   
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*5/8, "FULL REV", CM, 1,2, Black, Bone 
   
  RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*7/8, "RETURN", CM, 1,2, Black, Bone    
END SUB  
  ' -------------------------------------------------------------------
  ' subroutine called when SF screen is touched, to check if it has
  ' been touched on either the SINGLE, CONTINUOUS, 1/2 REVOLUTION,
  ' FULL REVOLUTION, FWD-THEN-REV, or RETURN buttons on the LCD
  ' (or not, in which case it just returns)
SUB CheckFn
  IntFlag = 0       ' first clear interrupt flag
  IF TOUCH(Y) < MM.VRes/4 THEN
    ExFlag = 1    ' not a valid touch, so just raise exit flag
    EXIT SUB        ' and return
  END IF
  IF TOUCH(X) < MM.HRes/2 THEN    ' if touch was on lower LH side
    SELECT CASE TOUCH(Y)
      CASE < MM.VRes/2  ' SINGLE button touched, so 
        Steps2Go = 1    ' set for a single step

      CASE MM.VRes/2 TO MM.VRes*3/4 ' HALF REV button touched, so 
        Steps2Go = (StpsperRev/2) * StpMult ' set Steps2Go to StpsperRev/2

      CASE > MM.VRes*3/4    ' FWD-REV button touched, so
        Steps2Go = StpsperRev * StpMult   ' set Steps2Go for 1 revolution
        SwingFlg = 1        ' and set SwingFlg for back-and-forth
     end select
  ELSE    ' touch must have been on lower RH side
    SELECT CASE TOUCH(Y)
      CASE < MM.VRes/2  ' CONTINUOUS button touched, so
        Steps2Go = 10000 * StpMult  ' set Stps2Go for 10000 full steps

      CASE MM.VRes/2 TO MM.VRes*3/4 ' FULL REV button touched, so 
        Steps2Go = StpsperRev * StpMult ' set Steps2Go equal to StpsperRev

      CASE > MM.VRes*3/4  ' RETURN button touched, so
        ExFlag = 1  ' just raise exit flag
        EXIT SUB    ' and return
    end select
  END IF
  ExFlag = 1
END SUB             ' before returning    
  ' -------------------------------------------------------------------  
  ' subroutine to show SelMode screen and allow mode selection
SUB SelMode
  ShowMSscreen    ' first go show the MS screen
  ExFlag = 0      ' clear exit flag
  DO
    IF IntFlag = 1 THEN   ' if screen touched
      CheckMode           ' go check & respond
      IF ExFlag = 1 THEN GOTO MainScrn
    END IF
  LOOP
END SUB  
  ' -------------------------------------------------------------------
  ' subroutine to show SelMode screen
SUB ShowMSscreen
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT MM.HRes/2, MM.VRes/16, "SELECT STEPPING", CM, 1, 2, White, Black
  TEXT MM.HRes/2, MM.VRes*3/16, "MODE:",CM, 1, 2, White, Black
  
  RBOX 4, MM.VRes/4, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes/4, MM.VRes*3/8, "FULL STEP", CM, 1, 2, Red, Bone

  RBOX MM.HRes/2, MM.VRes/4, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*3/8, "HALF STEP", CM, 1, 2, Red, Bone
   
  RBOX 4, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes/4, MM.VRes*5/8, "1/4 STEP", CM, 1, 2, Red, Bone 
   
  RBOX 4, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes/4, MM.VRes*7/8, "1/16 STEP", CM, 1, 2, Black, Bone 
   
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*5/8, "1/8 STEP", CM, 1,2, Black, Bone
   
  RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*7/8, "1/32 STEP", CM, 1,2, Black, Bone    
END SUB
  ' --------------------------------------------------------------------
  ' subroutine called when MS screen is touched, to check if it has
  ' been touched on either the FULL STEP, HALF STEP, 1/4 STEP, 1/8 STEP,
  ' 1/16 STEP or 1/32 STEP buttons on the LCD
  ' (or not, in which case it just returns)
SUB CheckMode
  IntFlag = 0       ' first clear interrupt flag
  IF TOUCH(Y) < MM.VRes/4 THEN
    ExFlag = 1      ' not a valid touch, so just raise exit flag
    EXIT SUB        ' and return
  END IF
  IF TOUCH(X) < MM.HRes/2 THEN    ' if touch was on lower LH side
    SELECT CASE TOUCH(Y)
      CASE < MM.VRes/2  ' FULL STEP button touched, so     
        StpMult = 1     ' set StpMult to 1 and
        PIN(21) = 0     ' set all 3 mode select pins to low
        PIN(18) = 0
        PIN(17) = 0
      CASE MM.VRes/2 TO MM.VRes*3/4 ' 1/4 STEP button touched, so
        StpMult = 4      ' set StpMult to 4 and
        PIN(21) = 0      ' set pins 21-18-17 for 1/4 stepping
        PIN(18) = 1
        PIN(17) = 0
      CASE > MM.VRes*3/4    ' 1/16 STEP button touched, so
        StpMult = 16    ' set StpMult to 16 and
        PIN(21) = 0     ' set pins 21-18-17 for 1/16 stepping
        PIN(18) = 0
        PIN(17) = 1
    end select
  ELSE    ' touch must have been on lower RH side
    SELECT CASE TOUCH(Y)
      CASE < MM.VRes/2  ' HALF STEP button touched, so
        StpMult = 2     ' set StpMult to 2 and
        PIN(21) = 1     ' set pins 21-18-17 for 1/2 stepping
        PIN(18) = 0
        PIN(17) = 0
      CASE MM.VRes/2 TO MM.VRes*3/4 ' 1/8 STEP button touched, so
        StpMult = 8     ' set StpMult to 8 and
        PIN(21) = 1     ' set pins 21-18-17 for 1/8 stepping
        PIN(18) = 1
        PIN(17) = 0
      CASE > MM.VRes*3/4  ' 1/32 STEP button touched, so
        StpMult = 32    ' set StpMult to 32 and
        PIN(21) = 1     ' set pins 21-18-17 for 1/32 stepping
        PIN(18) = 0
        PIN(17) = 1
    end select
  END IF
  ExFlag = 1        ' raise exit flag
END SUB             ' before returning    
  
  ' *******************************************************************
